#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.DAL.UnitTests
{

	[TestFixture]
	public class DAL_Patient : BaseTest 
	{
		
		private DataTable _dtPatient;
		private DataTable _dtPatientSpecimen;
		private DataTable _dtPatientChange;

		[SetUp]
		protected void SetUp() 
		{				
			this.BaseSetUp();
 
			if (this.RefreshData)
			{
				this._dtPatient = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Patient WHERE RecordStatusCode = 'A' AND MergeToPatient IS NULL ORDER BY LastUpdateDate DESC", true);

			

				System.Guid patientGuid = (System.Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];

				//*******PatientSpecimen
				this._dtPatientSpecimen = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientSpecimen WHERE SpecimenStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", false);
				if (this._dtPatientSpecimen.Rows.Count == 0)
				{
					DataTable dtPS = SprocHelper.GetInsertPatientSpecimenSprocTable(System.Guid.NewGuid(), patientGuid, DataCreator.GetNewSpecimenUid().ToString());
					new Common.StoredProcedure().TransactionalGetValue(Common.VbecsStoredProcs.InsertPatientSpecimen.StoredProcName, dtPS);
					this._dtPatientSpecimen = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientSpecimen WHERE SpecimenStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);
				}

				//*******PatientChange
				this._dtPatientChange = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientChange", false);
				if (this._dtPatientChange.Rows.Count == 0)
				{
					DataTable dtGuids = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 2 PatientGuid FROM Patient WHERE PatientSexCode IN ('M','F')");
					
					System.Guid fromPatientGuid = (System.Guid)dtGuids.Rows[0][TABLES.Patient.PatientGuid];
					System.Guid toPatientGuid = (System.Guid)dtGuids.Rows[1][TABLES.Patient.PatientGuid];

					DataTable dtPC = SprocHelper.GetHL7InsertPatientChange(System.Guid.NewGuid(), fromPatientGuid, toPatientGuid);
					new Common.StoredProcedure().TransactionalGetValue(Common.VbecsStoredProcs.HL7InsertPatientChange.StoredProcName, dtPC);
					
					this._dtPatientChange = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientChange", true);
				}

				this.RefreshData = false;
			}
		}

		[Test]
		public void Search_Pass()
		{
			string patientName = this._dtPatient.Rows[0][TABLES.Patient.PatientLastName].ToString();
			string patientSSN = this._dtPatient.Rows[0][TABLES.Patient.PatientSsn].ToString();
			string specimenUid = this._dtPatientSpecimen.Rows[0][TABLES.PatientSpecimen.SpecimenUid].ToString();

			DataTable dtTest = DAL.Patient.Search(patientName, patientSSN, specimenUid, UnitTestConfig.TestDivisionCode);
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Search1");

			dtTest = DAL.Patient.Search(string.Empty, patientSSN, specimenUid, UnitTestConfig.TestDivisionCode);
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Search2");

			dtTest = DAL.Patient.Search(patientName, string.Empty, specimenUid, UnitTestConfig.TestDivisionCode);
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Search3");

			dtTest = DAL.Patient.Search(patientName, patientSSN, string.Empty, UnitTestConfig.TestDivisionCode);
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Search4");

			Assert.AreEqual(16, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientGuid), TABLES.Patient.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.FinalMergeToPatient), TABLES.Patient.FinalMergeToPatient);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSsn), TABLES.Patient.PatientSsn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDob), TABLES.Patient.PatientDob);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSexCode), TABLES.Patient.PatientSexCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDobCode), TABLES.Patient.PatientDobCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDeathDate), TABLES.Patient.PatientDeathDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.AboRhChangeIndicator), TABLES.Patient.AboRhChangeIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ConversionBloodTypeCode), ARTIFICIAL.ConversionBloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ConversionRhFactorCode), ARTIFICIAL.ConversionRhFactorCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RowVersion), TABLES.Patient.RowVersion);
		}

		[Test]
		public void Search_Fail()
		{
			DataTable dtTest = DAL.Patient.Search("ABCDEFGHI", "ABCDEFGHI", "ABCDEFGHI", UnitTestConfig.TestDivisionCode);
			Assert.IsTrue(dtTest.Rows.Count == 0);
		}

		[Test]
		public void GetPatientDetails_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];

			DataTable dtTest = DAL.Patient.GetPatientDetails(unitGuid);

			Assert.AreEqual(28, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientGuid), TABLES.Patient.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientIcn), TABLES.Patient.PatientIcn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSsn), TABLES.Patient.PatientSsn);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.BloodTypeCode), TABLES.BloodUnitMedia.BloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitMedia.RhFactorCode), TABLES.BloodUnitMedia.RhFactorCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ConversionRhFactorCode), ARTIFICIAL.ConversionRhFactorCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ConversionBloodTypeCode), ARTIFICIAL.ConversionBloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.AboRhChangeIndicator), TABLES.Patient.AboRhChangeIndicator);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientNameSuffix), TABLES.Patient.PatientNameSuffix);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientNamePrefix), TABLES.Patient.PatientNamePrefix);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDob), TABLES.Patient.PatientDob);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDobCode), TABLES.Patient.PatientDobCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSexCode), TABLES.Patient.PatientSexCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientServiceBranch), TABLES.Patient.PatientServiceBranch);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.SsnPrefix), TABLES.Patient.SsnPrefix);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.SsnPseudoIndicator), TABLES.Patient.SsnPseudoIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDeathDate), TABLES.Patient.PatientDeathDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.MergeFromPatient), TABLES.Patient.MergeFromPatient);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.MergeToPatient), TABLES.Patient.MergeToPatient);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RecordStatusCode), TABLES.Patient.RecordStatusCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RowVersion), TABLES.Patient.RowVersion);
		}

		[Test]
		public void GetPatientDetails_Fail()
		{
			DataTable dtTest = DAL.Patient.GetPatientDetails(System.Guid.NewGuid());
			Assert.IsTrue(dtTest.Rows.Count == 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SaveSIsAndTRs_Pass()
		{
			//Call directly to Common.StoredProcedure.TransactionalGetValue -- See those unit tests
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SaveSIsAndTRs_Fail()
		{
			//Call directly to Common.StoredProcedure.TransactionalGetValue -- See those unit tests
		}

		[Test]
		public void GetPatientAboRHHistory_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			//
			//Insert Patient ABO/Rh results to ensure success.
			BOL.UnitTests.BOL_SpecimenTest.BuildABORhFromSpecimenTests(patientGuid, Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),"A","P");
			//
			DataTable dtTest = DAL.Patient.GetPatientAboRHHistory(patientGuid);
			Assert.IsTrue(dtTest.Rows.Count > 0,"Rows.Count");
			Assert.AreEqual(12,dtTest.Columns.Count,"Columns.Count");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.SpecimenTestGuid),"SpecimenTestGuid");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.PatientSpecimenGuid),"PatientSpecimenGuid");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.BloodTestTypeId),"BloodTestTypeId");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.TestDate),TABLES.BloodUnitTest.TestDate);
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.DivisionCode),TABLES.VamcDivision.DivisionCode);
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.TestResultId),"TestResultId");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.LastUpdateDate),"LastUpdateDate");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.TestTechId),TABLES.BloodUnitTest.TestTechId);
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.EntryTechId),"EntryTechId");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.AboRhChangeIndicator),"AboRhChangeIndicator");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.PatientSpecimen.SpecimenUid),"SpecimenUid");
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.RecordStatusCode),"RecordStatusCode");
		}

		[Test]
		public void GetPatientAboRHHistory_Fail()
		{
			DataTable dtTest = DAL.Patient.GetPatientAboRHHistory(System.Guid.NewGuid());
			Assert.IsTrue(dtTest.Rows.Count == 0);
		}

		[Test]
		public void GetPatientABORh_Pass()
		{
			// ABO/Rh generation happens according to BR_56.10
			//   1) No ABO/Rh results.
			//   2) Results from data conversion only.
			//   3) Two specimen results that are consistent.
			//   4a) Two specimen results that are NOT consistent (ABO inconsistent, Rh fine).
			//   4b) Two specimen results that are NOT consistent (Rh inconsistent, ABO fine).
			//   4c) Two specimen results that are NOT consistent (ABO and Rh both inconsistent).
			//   5) One specimen result that is consistent with Historical.
			//   6a) One specimen result is NOT consistent with Historical (ABO inconsistent, Rh fine).
			//   6b) One specimen result is NOT consistent with Historical (Rh inconsistent, ABO fine).
			//   6c) One specimen result is NOT consistent with Historical (ABO and Rh both inconsistent).
			//
			//	 *Note: If the patient has discrepant ABO/Rh results from the same
			//			or different specimens, the system displays the ABO/Rh
			//			results prior to the most recently recorded discrepancy.			

			// Situation #1: No results 
			// Historical: N/A (no results on file) <---  THIS is the one where ABO/Rh must come from.
			System.Guid patientGuid = System.Guid.NewGuid();

			DataTable dtPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			int retValue = new Common.StoredProcedure().TransactionalGetValue(Common.VbecsStoredProcs.InsertPatient.StoredProcName, dtPatient);

			Assert.AreEqual(0, retValue, "Patient Insert");

			UnitTestUtil.RunSQL("UPDATE Patient Set BloodTypeCode = 'A', RhFactorCode = 'P' WHERE PatientGuid = '" + patientGuid.ToString() + "'");

			DataRow dr = DAL.Patient.GetPatientABORh(patientGuid);

			Assert.AreEqual(dr[ARTIFICIAL.ABO].ToString().Trim(), "A", ARTIFICIAL.ABO);
			Assert.AreEqual( dr[ARTIFICIAL.Rh].ToString(), "P", ARTIFICIAL.Rh);
		}
		
		[Test]
		public void GetPatientABORh_Fail()
		{
			DataRow drTest = DAL.Patient.GetPatientABORh(System.Guid.NewGuid());

			Assert.AreEqual(true, drTest.IsNull(ARTIFICIAL.ABO), ARTIFICIAL.ABO);
		}

		[Test]
		public void GetAllSpecimensForPatient_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now;

			DataTable dtTest = DAL.Patient.GetAllSpecimensForPatient(unitGuid, startDate, endDate);
			Assert.IsTrue(dtTest.Rows.Count >= 0);
			Assert.IsTrue(dtTest.Columns.Count == 19);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetAllSpecimensForPatient_Fail()
		{
			System.Guid unitGuid = System.Guid.Empty;
			System.DateTime startDate = System.DateTime.Now.AddYears(-1);
			System.DateTime endDate = System.DateTime.Now;

			DataTable dtTest = DAL.Patient.GetAllSpecimensForPatient(unitGuid, startDate, endDate);
		}

		[Test]
		public void GetAllSpecimensForPatientCount_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];

			int intTest = DAL.Patient.GetAllSpecimensForPatientCount(unitGuid);
			Assert.IsTrue(intTest >= 0);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetAllSpecimensForPatientCount_Fail()
		{
			System.Guid unitGuid = System.Guid.Empty;

			int intTest = DAL.Patient.GetAllSpecimensForPatientCount(unitGuid);
		}

		[Test]
		public void GetRecentPatientABORhHistory_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];

			DataTable dtTest = DAL.Patient.GetRecentPatientAboRHHistory(unitGuid); 
			Assert.IsTrue(dtTest.Rows.Count >= 0);
			Assert.IsTrue(dtTest.Columns.Count == 11);
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.BloodTestTypeId));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.DivisionCode));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.SpecimenTestGuid));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.TestDate));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.TestComments));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.TestResultId));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.RowVersion));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.PatientSpecimen.SpecimenUid));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.PatientSpecimen.PatientSpecimenGuid));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.RhFactor.RhFactorText));
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.SpecimenTest.AboRhChangeIndicator));

		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetRecentPatientABORhHistory_Fail()
		{
			System.Guid unitGuid = System.Guid.Empty;

			DataTable dtTest = DAL.Patient.GetRecentPatientAboRHHistory(unitGuid);
		}

		[Test]
		public void JustifyAboRhChange_Pass()
		{
			this.RefreshData = true;

			// now, find a suitable patient guid ...
			System.Text.StringBuilder sql = new System.Text.StringBuilder("SELECT TOP 1(Patient.PatientGuid) FROM Patient ");
			sql.Append("LEFT OUTER JOIN PatientTreatment ON PatientTreatment.PatientGuid = Patient.PatientGuid ");
			sql.Append("LEFT OUTER JOIN PatientOrder ON PatientOrder.PatientTreatmentGuid = PatientTreatment.PatientTreatmentGuid ");
			sql.Append("LEFT OUTER JOIN OrderedTest ON PatientOrder.PatientOrderGuid = OrderedTest.PatientOrderGuid ");
			sql.Append("LEFT OUTER JOIN PatientSpecimen ON PatientSpecimen.PatientSpecimenGuid = OrderedTest.PatientSpecimenGuid ");
			sql.Append("JOIN SpecimenTest ST1 ON ST1.PatientSpecimenGuid = PatientSpecimen.PatientSpecimenGuid AND ST1.BloodTestTypeId = 18 ");
			sql.Append("JOIN SpecimenTest ST2 ON ST2.PatientSpecimenGuid = PatientSpecimen.PatientSpecimenGuid AND ST2.BloodTestTypeId = 21 ");
			sql.Append("WHERE ST1.RecordStatusCode = 'A' AND ST2.RecordStatusCode = 'A'");

			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable(sql.ToString(), true);

			Guid patientGuid = (Guid)dtTemp.Rows[0][TABLES.Patient.PatientGuid];

			// Load patient data
			BOL.Patient patient = new BOL.Patient(patientGuid);

			// Get latest test results
			DataTable dtSecimens = DAL.Patient.GetRecentPatientAboRHHistory(patientGuid);

			DataTable dtPatient = new DataTable();
			dtPatient.Columns.Add(TABLES.Patient.PatientGuid, typeof(System.Guid));
			dtPatient.Columns.Add(TABLES.Patient.AboRhChangeIndicator, typeof(string));
			dtPatient.Columns.Add(TABLES.Patient.LastUpdateUser, typeof(string));
			dtPatient.Columns.Add(TABLES.Patient.RowVersion, typeof(byte[]));

			DataRow drPatient = dtPatient.NewRow();
			drPatient[TABLES.Patient.PatientGuid] = patient.PatientGuid;
			drPatient[TABLES.Patient.AboRhChangeIndicator] = 1;
			drPatient[TABLES.Patient.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drPatient[TABLES.Patient.RowVersion] = patient.RowVersion;
			dtPatient.Rows.Add(drPatient);
			// **********************************************************************************************************************
			DataTable dtAboSpecimenTest = new DataTable();
			dtAboSpecimenTest.Columns.Add(TABLES.SpecimenTest.SpecimenTestGuid, typeof(System.Guid));
			dtAboSpecimenTest.Columns.Add(TABLES.SpecimenTest.TestResultId, typeof(string));
			dtAboSpecimenTest.Columns.Add(TABLES.SpecimenTest.AboRhChangeIndicator , typeof(int));
			dtAboSpecimenTest.Columns.Add(TABLES.SpecimenTest.AboRhChangeJustification, typeof(string));
			dtAboSpecimenTest.Columns.Add(TABLES.SpecimenTest.LastUpdateUser, typeof(string));
			dtAboSpecimenTest.Columns.Add(TABLES.SpecimenTest.RowVersion, typeof(byte[]));

			DataRow drAboSpecimenTest = dtAboSpecimenTest.NewRow();
			drAboSpecimenTest[TABLES.SpecimenTest.SpecimenTestGuid] = dtSecimens.Rows[0][TABLES.SpecimenTest.SpecimenTestGuid];
			drAboSpecimenTest[TABLES.SpecimenTest.TestResultId] = "AB";
			drAboSpecimenTest[TABLES.SpecimenTest.AboRhChangeIndicator] = 1;
			drAboSpecimenTest[TABLES.SpecimenTest.AboRhChangeJustification] = "test justify abo " + System.DateTime.Now;
			drAboSpecimenTest[TABLES.SpecimenTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drAboSpecimenTest[TABLES.SpecimenTest.RowVersion] = dtSecimens.Rows[0][TABLES.SpecimenTest.RowVersion];
			dtAboSpecimenTest.Rows.Add(drAboSpecimenTest);
			// **********************************************************************************************************************
			DataTable dtRhSpecimenTest = new DataTable();
			dtRhSpecimenTest.Columns.Add(TABLES.SpecimenTest.SpecimenTestGuid, typeof(System.Guid));
			dtRhSpecimenTest.Columns.Add(TABLES.SpecimenTest.TestResultId, typeof(string));
			dtRhSpecimenTest.Columns.Add(TABLES.SpecimenTest.AboRhChangeIndicator , typeof(int));
			dtRhSpecimenTest.Columns.Add(TABLES.SpecimenTest.AboRhChangeJustification, typeof(string));
			dtRhSpecimenTest.Columns.Add(TABLES.SpecimenTest.LastUpdateUser, typeof(string));
			dtRhSpecimenTest.Columns.Add(TABLES.SpecimenTest.RowVersion, typeof(byte[]));

			DataRow drRhSpecimenTest = dtRhSpecimenTest.NewRow();
			drRhSpecimenTest[TABLES.SpecimenTest.SpecimenTestGuid] = dtSecimens.Rows[1][TABLES.SpecimenTest.SpecimenTestGuid];
			drRhSpecimenTest[TABLES.SpecimenTest.TestResultId] = "I";
			drRhSpecimenTest[TABLES.SpecimenTest.AboRhChangeIndicator] = 1;
			drRhSpecimenTest[TABLES.SpecimenTest.AboRhChangeJustification] = "test justify rh " + System.DateTime.Now;
			drRhSpecimenTest[TABLES.SpecimenTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drRhSpecimenTest[TABLES.SpecimenTest.RowVersion] = dtSecimens.Rows[1][TABLES.SpecimenTest.RowVersion];
			dtRhSpecimenTest.Rows.Add(drRhSpecimenTest);
			// **********************************************************************************************************************
			System.Guid [] patientGuids = new Guid[1];
			patientGuids[0] = patient.PatientGuid;
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.Justification);
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.Justification, patientGuids, Common.WorkloadTransactionType.Miscellanious, false);
			Assert.IsTrue((DAL.Patient.JustifyAboRHChange(dtPatient, dtAboSpecimenTest, dtRhSpecimenTest, Common.UpdateFunction.UnitTests, dtWorkloadEvent)));
		}

		[Test]
		public void JustifyAboRhChange_Fail()
		{
			DataTable dtPatient = SprocHelper.GetUpdatePatientAboRHChangeIndicatorSprocTable();
			DataTable dtAboTest = SprocHelper.GetUpdateSpecimenTestSprocTable();
			DataTable dtRhTest = SprocHelper.GetUpdateSpecimenTestSprocTable();
			DataTable dtEmpty = new DataTable();

			dtPatient.Rows[0][TABLES.Patient.PatientGuid] = this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];
			dtPatient.Rows[0][TABLES.Patient.RowVersion] = this._dtPatient.Rows[0][TABLES.Patient.RowVersion];
			System.Guid [] patientGuids = new Guid [1];
			patientGuids[0] = (System.Guid) this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];

			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.Justification);
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.Justification, patientGuids, Common.WorkloadTransactionType.Miscellanious, false);

			try
			{
				Assert.IsTrue(DAL.Patient.JustifyAboRHChange(dtEmpty, dtAboTest, dtRhTest, Common.UpdateFunction.UnitTests, dtWorkloadEvent));
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(Common.RowVersionException)
			{
				Assert.IsTrue(true);
			}

			try
			{
				Assert.IsTrue(DAL.Patient.JustifyAboRHChange(dtPatient, dtEmpty, dtRhTest, Common.UpdateFunction.UnitTests, dtWorkloadEvent));
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(Common.RowVersionException)
			{
				Assert.IsTrue(true);
			}
			
			try
			{
				Assert.IsTrue(DAL.Patient.JustifyAboRHChange(dtPatient, dtAboTest, dtEmpty, Common.UpdateFunction.UnitTests, dtWorkloadEvent));
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(Common.RowVersionException)
			{
				Assert.IsTrue(true);
			}

		}

		[Test]
		public void UpdatePatientChangeStatus_Pass()
		{
			this.RefreshData = true;

			System.Guid unitGuid = (System.Guid)this._dtPatientChange.Rows[0][TABLES.PatientChange.PatientChangeGuid];
			Common.PatientChangeStatus changeStatus = Common.PatientChangeStatus.InProcess;
			byte[] rowVersion = (byte[])this._dtPatientChange.Rows[0][TABLES.PatientChange.RowVersion];

			bool retValue = DAL.Patient.UpdatePatientChangeStatus(unitGuid, changeStatus, string.Empty, rowVersion, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(retValue);
		}

		[Test]
		public void UpdatePatientChangeStatus_Fail()
		{
			this.RefreshData = true;

			System.Guid unitGuid = (System.Guid)this._dtPatientChange.Rows[0][TABLES.PatientChange.PatientChangeGuid];
			Common.PatientChangeStatus changeStatus = Common.PatientChangeStatus.InProcess;
			byte[] rowVersion = new byte[8];

			try
			{
				DAL.Patient.UpdatePatientChangeStatus(System.Guid.Empty, changeStatus, string.Empty, rowVersion, Common.UpdateFunction.UnitTests);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}

			try
			{
				DAL.Patient.UpdatePatientChangeStatus(unitGuid, Common.PatientChangeStatus.Unknown, string.Empty, rowVersion, Common.UpdateFunction.UnitTests);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}

			try
			{
				DAL.Patient.UpdatePatientChangeStatus(unitGuid, changeStatus, string.Empty, rowVersion, Common.UpdateFunction.UnitTests);
				Assert.IsTrue(false, AssertionFailReasons.RowVersionExpected);
			}
			catch(Common.RowVersionException)
			{
				Assert.IsTrue(true);
			}
		}

		[Test]
		public void GetTreatingSpecialtyList_Pass()
		{
			DataTable dtTest = DAL.Patient.GetTreatingSpecialtyList();
			Assert.IsTrue(dtTest.Rows.Count > 0);
			Assert.IsTrue(dtTest.Columns.Count == 1);
			Assert.IsTrue(dtTest.Columns.Contains(TABLES.TreatingSpecialty.TreatingSpecialtyName));
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetTreatingSpecialtyList_Fail()
		{
		}


		[Test]
		public void GetPatientAboRhJustificationReason_Pass()
		{
			DataRow drPatient = UnitTestUtil.RunSQLGetDataRow( "select top 1 Patient.PatientGuid from Patient" );
			Guid patientGuid = (Guid)drPatient[Common.VbecsTables.Patient.PatientGuid];

			Guid patientTreatmentGuid = Guid.NewGuid();
			string insertPatientTreatment = "INSERT INTO dbo.PatientTREATMENT (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'UnitTestUser',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientOrderGuid = Guid.NewGuid();
			string insertPatientOrder = "INSERT INTO dbo.PatientORDER (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'UnitTestUser',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedTestGuid = Guid.NewGuid();
			string insertOrderedTest = "INSERT INTO ORDEREDTEST (";
			insertOrderedTest += TABLES.OrderedTest.OrderedTestGuid + ",";
			insertOrderedTest += TABLES.OrderedTest.PatientOrderGuid + ",";
			insertOrderedTest += TABLES.OrderedTest.OrderableTestId + ",";
			insertOrderedTest += TABLES.OrderedTest.OrderStatusCode + ",";
			insertOrderedTest += TABLES.OrderedTest.TaskStatusCode + ",";
			insertOrderedTest += TABLES.OrderedTest.DivisionCode + ",";
			insertOrderedTest += TABLES.OrderedTest.LastUpdateDate + ",";
			insertOrderedTest += TABLES.OrderedTest.LastUpdateUser + ",";
			insertOrderedTest += TABLES.OrderedTest.LastUpdateFunctionId + ")";
			insertOrderedTest += "VALUES (";
			insertOrderedTest += "'" + orderedTestGuid.ToString() + "',";
			insertOrderedTest += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedTest += "'1',";
			insertOrderedTest += "'A',";
			insertOrderedTest += "'P',";
			insertOrderedTest += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedTest += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedTest += "'UnitTestUser',";
			insertOrderedTest += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientSpecimenGuid = Guid.NewGuid();
			string insertPatientSpecimen = "INSERT INTO dbo.PatientSPECIMEN (";
			insertPatientSpecimen += TABLES.PatientSpecimen.PatientSpecimenGuid + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.PatientGuid + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenCollectionDate + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenStatusCode + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.UnacceptableSpecimenIndicator + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenUid + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.DivisionCode + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.LastUpdateDate + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.LastUpdateUser + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.LastUpdateFunctionId + ")";
			insertPatientSpecimen += "VALUES (";
			insertPatientSpecimen += "'" + patientSpecimenGuid.ToString() + "',";
			insertPatientSpecimen += "'" + patientGuid.ToString() + "',";
			insertPatientSpecimen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientSpecimen += "'A',";
			insertPatientSpecimen += "'0',";
			insertPatientSpecimen += "'" + DataCreator.GetNewSpecimenUid() + "',";
			insertPatientSpecimen += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertPatientSpecimen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientSpecimen += "'UnitTestUser',";
			insertPatientSpecimen += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid specimenTestGuid = Guid.NewGuid();
			string insertSpecimenTest = "INSERT INTO SPECIMENTEST (";
			insertSpecimenTest += TABLES.SpecimenTest.SpecimenTestGuid + ",";
			insertSpecimenTest += TABLES.SpecimenTest.PatientSpecimenGuid + ",";
			insertSpecimenTest += TABLES.SpecimenTest.OrderedTestGuid + ",";
			insertSpecimenTest += TABLES.SpecimenTest.BloodTestTypeId + ",";
			insertSpecimenTest += TABLES.SpecimenTest.TestResultId + ",";
			insertSpecimenTest += TABLES.SpecimenTest.AboRhChangeIndicator + ",";
			insertSpecimenTest += TABLES.SpecimenTest.AboRhChangeJustification + ",";
			insertSpecimenTest += TABLES.SpecimenTest.RecordStatusCode + ",";
			insertSpecimenTest += TABLES.PatientSpecimen.DivisionCode + ",";
			insertSpecimenTest += TABLES.PatientSpecimen.LastUpdateDate + ",";
			insertSpecimenTest += TABLES.PatientSpecimen.LastUpdateUser + ",";
			insertSpecimenTest += TABLES.PatientSpecimen.LastUpdateFunctionId + ")";
			insertSpecimenTest += "VALUES (";
			insertSpecimenTest += "'" + specimenTestGuid.ToString() + "',";
			insertSpecimenTest += "'" + patientSpecimenGuid.ToString() + "',";
			insertSpecimenTest += "'" + orderedTestGuid.ToString() + "',";
			insertSpecimenTest += "'18',";
			insertSpecimenTest += "'A',";
			insertSpecimenTest += "1,";
			insertSpecimenTest += "'blah blah',";
			insertSpecimenTest += "'A',";
			insertSpecimenTest += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertSpecimenTest += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertSpecimenTest += "'UnitTestUser',";
			insertSpecimenTest += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedTest);
			UnitTestUtil.RunSQL(insertPatientSpecimen);
			UnitTestUtil.RunSQL(insertSpecimenTest);
			UnitTestUtil.RunSQL("UPDATE PATIENT SET AboRhChangeIndicator = 1 WHERE PatientGuid = '" + patientGuid.ToString() + "'");

			DataRow dr = UnitTestUtil.RunSQLGetDataRow( "select top 1 Patient.PatientGuid from Patient join PatientSpecimen on Patient.PatientGuid = PatientSpecimen.PatientGuid join SpecimenTest on PatientSpecimen.PatientSpecimenGuid = SpecimenTest.PatientSpecimenGuid where Patient.AboRhChangeIndicator = 1 AND SpecimenTest.AboRhChangeIndicator = 1" );
			patientGuid = (Guid) dr[Common.VbecsTables.Patient.PatientGuid];
			//
			DataRow dtRow = DAL.Patient.GetPatientAboRhJustificationReason(patientGuid);
			Assert.IsTrue(dtRow.Table.Columns.Contains(Common.VbecsTables.SpecimenTest.AboRhChangeJustification));
		}
		[Test]
		public void GetPatientAboRhJustificationReason_Fail()
		{
			DataRow dtRow = DAL.Patient.GetPatientAboRhJustificationReason(System.Guid.NewGuid());

			Assert.IsNull(dtRow);
		}

		[Test]
		public void GetEmptyPatientTableSchema_Pass()
		{
			DataTable dtInsert = DAL.Patient.GetEmptyPatientTableSchema(false);
			Assert.AreEqual(14,dtInsert.Columns.Count,"InsertColumnCount");
			//
			DataTable dtUpdate = DAL.Patient.GetEmptyPatientTableSchema(true);
			Assert.AreEqual(15,dtUpdate.Columns.Count,"UpdateColumnCount");
			Assert.IsTrue(dtUpdate.Columns.Contains(TABLES.Patient.RowVersion),"ContainsRowVersion");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetEmptyPatientTableSchema_Fail()
		{
		}

		[Test]
		public void GetEmptyPatientTreatmentTableSchema_Pass()
		{
			DataTable dtInsert = DAL.Patient.GetEmptyPatientTreatmentTableSchema(false);
			Assert.AreEqual(6,dtInsert.Columns.Count,"InsertColumnCount");
			//
			DataTable dtUpdate = DAL.Patient.GetEmptyPatientTreatmentTableSchema(true);
			Assert.AreEqual(7,dtUpdate.Columns.Count,"UpdateColumnCount");
			Assert.IsTrue(dtUpdate.Columns.Contains(TABLES.PatientTreatment.RowVersion),"ContainsRowVersion");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetEmptyPatientTreatmentTableSchema_Fail()
		{
		}

		[Test]
		public void GetPatientDetails_IdorIcn_Pass()
		{
			DataTable dtPatientId = DAL.Patient.GetPatientDetails(Convert.ToInt64(_dtPatient.Rows[0][TABLES.Patient.VistaPatientId]),string.Empty);
			Assert.AreEqual(1,dtPatientId.Rows.Count,"PatientIdRowCount");
			Assert.AreEqual(Convert.ToInt64(_dtPatient.Rows[0][TABLES.Patient.VistaPatientId]),Convert.ToInt64(dtPatientId.Rows[0][TABLES.Patient.VistaPatientId]),"PatientId");
			//
			DataRow drIcn = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 "+TABLES.Patient.PatientIcn+" FROM "+TABLES.Patient.TableName+" WHERE "+TABLES.Patient.PatientIcn+" IS NOT NULL");
			DataTable dtPatientIcn = DAL.Patient.GetPatientDetails(Int32.MinValue,drIcn[TABLES.Patient.PatientIcn].ToString());
			Assert.AreEqual(1,dtPatientIcn.Rows.Count,"PatientIcnRowCount");
			Assert.AreEqual(drIcn[TABLES.Patient.PatientIcn].ToString(),dtPatientIcn.Rows[0][TABLES.Patient.PatientIcn].ToString(),"PatientIcn");
		}

		[Test]
		public void GetPatientDetails_IdorIcn_Fail()
		{
			DataTable dtPatientId = DAL.Patient.GetPatientDetails(Int32.MinValue,string.Empty);
			Assert.AreEqual(0,dtPatientId.Rows.Count,"PatientIdRowCount");
			//
			DataTable dtPatientIcn = DAL.Patient.GetPatientDetails(Int32.MinValue,"INVALID ICN");
			Assert.AreEqual(0,dtPatientIcn.Rows.Count,"PatientIcnRowCount");
		}

		[Test]
		public void GetAboRhTestHistory_Pass()
		{
			System.Guid patientGuid = DataCreator.GetRecordGuid("SELECT TOP 1 PatientGuid FROM Patient WHERE PatientGuid IN (SELECT PS.PatientGuid FROM PatientSpecimen PS JOIN SpecimenTest ST ON PS.PatientSpecimenGuid = ST.PatientSpecimenGuid WHERE ST.BloodTestTypeId IN (18,21,65,68))");
			
			DataTable dtAborh = DAL.Patient.GetAboRHTestHistory(patientGuid, DateTime.Now);
			Assert.IsTrue(dtAborh.Rows.Count>0,"RowCount");
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetAboRhTestHistory_Fail()
		{
			DataTable dtAborh = DAL.Patient.GetAboRHTestHistory(Guid.Empty,DateTime.Now);
			Assert.AreEqual(0,dtAborh.Rows.Count,"RowCount");
		}

		// **************************************************************************************************************
		// ****	PENDING PATIENT MERGES ***
		// *******************************

		[Test]
		public void GetPendingPatientMergesDetails_Pass()
		{
			BOL.Patient mergeFromPatient = new BOL.Patient(CreatePatientTestData(new BOL.AboRh(Common.ABO.A, Common.RH.Positive)));
			BOL.Patient mergeToPatient = new BOL.Patient(CreatePatientTestData(new BOL.AboRh(Common.ABO.NA, Common.RH.NotProvided)));
			//
			InsertMergePatientTestData(mergeFromPatient, mergeToPatient);
			//
			DataTable dtTest = DAL.Patient.GetPendingPatientMergesDetails(UnitTestConfig.TestDivisionCode);		
			
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "RowCount");
			Assert.AreEqual(18, dtTest.Columns.Count, "Columns");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientGuid), TABLES.PatientChange.FromPatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.MergeFromVistaPatientId), TABLES.PatientChange.MergeFromVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.FromDisplayVistaPatientId), ARTIFICIAL.FromDisplayVistaPatientId);
			Assert.AreEqual(true, (dtTest.Columns.Contains(TABLES.PatientChange.FromPatientLastName) || dtTest.Columns.Contains(TABLES.Patient.PatientLastName)), TABLES.PatientChange.FromPatientLastName);
			Assert.AreEqual(true, (dtTest.Columns.Contains(TABLES.PatientChange.FromPatientFirstName) || dtTest.Columns.Contains(TABLES.Patient.PatientFirstName)), TABLES.PatientChange.FromPatientFirstName);

			Assert.AreEqual(true, (dtTest.Columns.Contains(TABLES.PatientChange.FromPatientMIddleName) || dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName)), TABLES.PatientChange.FromPatientMIddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientGuid), TABLES.PatientChange.ToPatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.MergeToVistaPatientId), TABLES.PatientChange.MergeToVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ToDisplayVistaPatientId), ARTIFICIAL.ToDisplayVistaPatientId);
			Assert.AreEqual(true, (dtTest.Columns.Contains(TABLES.PatientChange.ToPatientLastName) || dtTest.Columns.Contains(TABLES.Patient.PatientLastName)), TABLES.PatientChange.ToPatientLastName);

			Assert.AreEqual(true, (dtTest.Columns.Contains(TABLES.PatientChange.ToPatientFirstName) || dtTest.Columns.Contains(TABLES.Patient.PatientFirstName)), TABLES.PatientChange.ToPatientFirstName);
			Assert.AreEqual(true, (dtTest.Columns.Contains(TABLES.PatientChange.ToPatientMIddleName) || dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName)), TABLES.PatientChange.ToPatientMIddleName);
			Assert.AreEqual(true, (dtTest.Columns.Contains(TABLES.PatientChange.ToPatientSexCode)), TABLES.PatientChange.ToPatientSexCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeGuid), TABLES.PatientChange.PatientChangeGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeStatusCode), TABLES.PatientChange.PatientChangeStatusCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.NotificationReceivedDate), TABLES.PatientChange.NotificationReceivedDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.LastUpdateDate), TABLES.PatientChange.LastUpdateDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.RowVersion), TABLES.PatientChange.RowVersion);
			//
			Guid patientChangeGuid = (Guid)dtTest.Rows[0][TABLES.PatientChange.PatientChangeGuid];
			UnitTestUtil.RunSQL(string.Concat("DELETE FROM PatientChange WHERE PatientChangeGuid = '", patientChangeGuid, "'"));
		}

		[Test]
		public void GetPendingPatientMergesDetails_Fail()
		{
			try
			{
				DataTable dtTest = DAL.Patient.GetPendingPatientMergesDetails(string.Empty);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
			
			try
			{
				DataTable dtTest = DAL.Patient.GetPendingPatientMergesDetails(null);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		[Test]
		public void GetPendingPatientMerges_Pass()
		{
			BOL.Patient mergeFromPatient = new BOL.Patient(CreatePatientTestData(new BOL.AboRh(Common.ABO.A, Common.RH.Positive)));
			BOL.Patient mergeToPatient = new BOL.Patient(CreatePatientTestData(new BOL.AboRh(Common.ABO.NA, Common.RH.NotProvided)));
			//
			Guid patientChangeGuid = InsertMergePatientTestData(mergeFromPatient, mergeToPatient);
			//
			DataTable dtTest = DAL.Patient.GetPendingPatientMerges(UnitTestConfig.TestDivisionCode);		
			Assert.IsTrue(dtTest.Rows.Count == 1);
			Assert.IsTrue(dtTest.Columns.Count == 1);
			Assert.IsTrue(dtTest.Columns.Contains(ARTIFICIAL.PendingPatientMergesCount));
			Assert.IsTrue(Convert.ToInt32(dtTest.Rows[0][ARTIFICIAL.PendingPatientMergesCount]) > 0);
			//
			UnitTestUtil.RunSQL(string.Concat("DELETE FROM PatientChange WHERE PatientChangeGuid = '", patientChangeGuid, "'"));
		}

		[Test]
		public void GetPendingPatientMerges_Fail()
		{
			try
			{
				DataTable dtTest = DAL.Patient.GetPendingPatientMerges(string.Empty);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
			
			try
			{
				DataTable dtTest = DAL.Patient.GetPendingPatientMerges(null);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		// **************************************************************************************************************
		// ****	DECEASED PATIENTS ***
		// **************************

		[Test]
		public void GetRecentlyDeceasedPatientDetails_Pass()
		{
			DataTable dtTest = DAL.Patient.GetRecentlyDeceasedPatientDetails(UnitTestConfig.TestDivisionCode);		

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "RowCount");
			Assert.AreEqual(13, dtTest.Columns.Count, "Columns");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeGuid), TABLES.PatientChange.PatientChangeGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientGuid), TABLES.PatientChange.FromPatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientLastName), TABLES.PatientChange.ToPatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientFirstName), TABLES.PatientChange.ToPatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientMIddleName), TABLES.PatientChange.ToPatientMIddleName);

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ToDisplayVistaPatientId), ARTIFICIAL.ToDisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientDob), TABLES.PatientChange.ToPatientDob);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientDeathDate), TABLES.PatientChange.ToPatientDeathDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.NotificationReceivedDate), TABLES.PatientChange.NotificationReceivedDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.RowVersion), TABLES.PatientChange.RowVersion);
		
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeStatusCode), TABLES.PatientChange.PatientChangeStatusCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ActiveDivisionsAlertViewed), TABLES.PatientChange.ActiveDivisionsAlertViewed);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.DivisionsWithActiveOrders), ARTIFICIAL.DivisionsWithActiveOrders);
		}

		[Test]
		public void GetRecentlyDeceasedPatientDetails_Fail()
		{
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyDeceasedPatientDetails(string.Empty);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
			
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyDeceasedPatientDetails(null);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		[Test]
		public void GetRecentlyDeceasedPatients_Pass()
		{
			DataTable dtTest = DAL.Patient.GetRecentlyDeceasedPatients(UnitTestConfig.TestDivisionCode);
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
			Assert.AreEqual(5, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeGuid), TABLES.PatientChange.PatientChangeGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientGuid), TABLES.PatientChange.FromPatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ActiveDivisionsAlertViewed), TABLES.PatientChange.ActiveDivisionsAlertViewed);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.DivisionsWithActiveOrders), ARTIFICIAL.DivisionsWithActiveOrders);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.RowVersion), TABLES.PatientOrder.RowVersion);
		}

		[Test]
		public void GetRecentlyDeceasedPatients_Fail()
		{
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyDeceasedPatients(string.Empty);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
			
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyDeceasedPatients(null);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		// **************************************************************************************************************
		// ****	UPDATED PATIENTS ***
		// *************************

		[Test]
		public void GetRecentlyUpdatedPatientDetails_Pass()
		{
			DataTable dtTest = DAL.Patient.GetRecentlyUpdatedPatientDetails(UnitTestConfig.TestDivisionCode);		

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
			Assert.AreEqual(23, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeGuid), TABLES.PatientChange.PatientChangeGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeStatusCode), TABLES.PatientChange.PatientChangeStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientGuid), TABLES.PatientChange.FromPatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientLastName), TABLES.PatientChange.FromPatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientFirstName), TABLES.PatientChange.FromPatientFirstName);

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.FromPatientMiddleName), ARTIFICIAL.FromPatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.FromDisplayVistaPatientId), ARTIFICIAL.FromDisplayVistaPatientId);
			Assert.AreEqual(true,dtTest.Columns.Contains(TABLES.PatientChange.FromPatientIcn), TABLES.PatientChange.FromPatientIcn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientDob), TABLES.PatientChange.FromPatientDob);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientSexCode), TABLES.PatientChange.FromPatientSexCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromSsnPseudoIndicator), TABLES.PatientChange.FromSsnPseudoIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientLastName), TABLES.PatientChange.ToPatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientFirstName), TABLES.PatientChange.ToPatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ToPatientMiddleName), ARTIFICIAL.ToPatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.ToDisplayVistaPatientId), ARTIFICIAL.ToDisplayVistaPatientId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientIcn), TABLES.PatientChange.ToPatientIcn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientDob), TABLES.PatientChange.ToPatientDob);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToPatientSexCode), TABLES.PatientChange.ToPatientSexCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ToSsnPseudoIndicator), TABLES.PatientChange.ToSsnPseudoIndicator);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.NotificationReceivedDate), TABLES.PatientChange.NotificationReceivedDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ActiveDivisionsAlertViewed), TABLES.PatientChange.ActiveDivisionsAlertViewed);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.DivisionsWithActiveOrders), ARTIFICIAL.DivisionsWithActiveOrders);
		
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.RowVersion), TABLES.PatientChange.RowVersion);
		}

		[Test]
		public void GetRecentlyUpdatedPatientDetails_Fail()
		{
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyUpdatedPatientDetails(string.Empty);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
			
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyUpdatedPatientDetails(null);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		[Test]
		public void GetRecentlyUpdatedPatients_Pass()
		{
			DataTable dtTest = DAL.Patient.GetRecentlyUpdatedPatients(UnitTestConfig.TestDivisionCode);		
			
			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
			Assert.AreEqual(5, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.PatientChangeGuid), TABLES.PatientChange.PatientChangeGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.FromPatientGuid), TABLES.PatientChange.FromPatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.ActiveDivisionsAlertViewed), TABLES.PatientChange.ActiveDivisionsAlertViewed);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.DivisionsWithActiveOrders), ARTIFICIAL.DivisionsWithActiveOrders);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientChange.RowVersion), TABLES.PatientChange.RowVersion);
		}

		[Test]
		public void GetRecentlyUpdatedPatients_Fail()
		{
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyUpdatedPatients(string.Empty);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
			
			try
			{
				DataTable dtTest = DAL.Patient.GetRecentlyUpdatedPatients(null);
				Assert.IsTrue(false, AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true);
			}
		}

		#region Private Methods - no UnitTests

		/// <summary>
		/// </summary>
		/// <returns>Guid</returns>
		private Guid CreatePatientTestData(BOL.AboRh aboRh)
		{
			Guid patientGuid = Guid.NewGuid();

			DataTable dtInsPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			bool retValue = BOL.Patient.InsertPatient(dtInsPatient.Rows[0], Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, retValue, "PatientInsert");

			UnitTestUtil.RunSQL("UPDATE Patient SET BloodTypeCode = '" + aboRh.Abo.ToString() + "', RhFactorCode = '" + aboRh.RHFactorCode + "' WHERE PatientGuid = '" + patientGuid.ToString() + "'");

			return patientGuid;
		}

		/// <summary>
		/// Inserts merge test data into PatientChange
		/// </summary>
		/// <returns>Row change count</returns>
		private Guid InsertMergePatientTestData(BOL.Patient mergeFromPatient, BOL.Patient mergeToPatient)
		{		
			StringBuilder query = new StringBuilder();
			// 
			Guid patientChangeGuid = Guid.NewGuid();
			//
			query.Remove(0, query.Length);
			query.Append("INSERT INTO PatientChange (PatientChangeGuid, FromPatientGuid, ToPatientGuid, MergeFromVistaPatientId, ");
			query.Append("MergeToVistaPatientId, PatientChangeStatusCode, NotificationReceivedDate, LastUpdateFunctionId, LastUpdateUser, LastUpdateDate)");
			query.Append("VALUES ('");
			query.Append(patientChangeGuid).Append("', '");
			query.Append(mergeFromPatient.PatientGuid).Append("', '");
			query.Append(mergeToPatient.PatientGuid).Append("', '");
			query.Append(mergeFromPatient.VistaPatientId).Append("', '");
			query.Append(mergeToPatient.VistaPatientId).Append("', 'P', GetUtcDate(), 0, 'UNITTEST', GetUtcDate())");
			//
			int rowCount = UnitTestUtil.RunSQL(query.ToString());
			//
			Assert.IsTrue(rowCount > 0, "UNABLE TO INSERT PATIENT CHANGE DATA!");
			//
			return patientChangeGuid;
		}

		/// <summary>
		/// Removes any new patient data so the unit tests can be run over an over
		/// </summary>
		/// <param name="patientGuid"></param>
		private void DeleteNewPatientRecords(System.Guid patientGuid)
		{
			string sql1 = "DELETE FROM PatientChange WHERE FromPatientGuid = '" + patientGuid.ToString() + "'";
			string sql2 = "DELETE FROM PatientTreatment WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			string sql3 = "DELETE FROM Patient WHERE PatientGuid = '" + patientGuid.ToString() + "'";

			UnitTestUtil.RunSQLInTransaction(sql1, sql2, sql3);
		}

		private void DeleteMergedPatientRecords(System.Guid fromPatientGuid, System.Guid toPatientGuid)
		{
			string sql1 = "DELETE FROM PatientChange WHERE FromPatientGuid = '" + fromPatientGuid.ToString() + "'";
			string sql2 = "DELETE FROM PatientTreatment WHERE PatientGuid IN ('" + fromPatientGuid.ToString() + "','"+ toPatientGuid.ToString() + "')";
			string sql3 = "DELETE FROM Patient WHERE PatientGuid IN ('" + fromPatientGuid.ToString() + "','"+ toPatientGuid.ToString() + "')";

			UnitTestUtil.RunSQLInTransaction(sql1, sql2, sql3);
		}

		/// <summary>
		/// Inserts merge test data into PatientChange
		/// </summary>
		/// <returns>Row change count</returns>
		private Guid [] InsertMergePatientTestData(ref string fromVistaPatientId)
		{		
			this.RefreshData = true;

			Guid [] returnGuids = new Guid[3];
			//
			// from-Patient
			returnGuids[0] = Guid.NewGuid();
			// to-Patient
			returnGuids[1] = Guid.NewGuid();
			// Patient Change
			returnGuids[2] = Guid.NewGuid();
			//
			DataTable patientSsnTable = UnitTestUtil.RunSQLGetDataTable("SELECT DISTINCT PatientSsn FROM Patient ORDER BY PatientSsn DESC");
			DataColumn [] patientSsnKeys = new DataColumn []{ patientSsnTable.Columns[TABLES.Patient.PatientSsn] };
			patientSsnTable.PrimaryKey = patientSsnKeys;
			//
			DataRow vistaIdRow = UnitTestUtil.RunSQLGetDataRow("SELECT MAX(VistaPatientId) AS VistaPatientId FROM Patient");
			long maxVistaId = Convert.ToInt64( vistaIdRow[TABLES.Patient.VistaPatientId] );
			long mergeFromVistaId = maxVistaId + 1;
			long mergeToVistaId = mergeFromVistaId + 1;
			//
			int maxSsn = 999999999;
			int minSsn = 100000000;
			int mergeFromPatientSsn = minSsn;
			int mergeToPatientSsn = minSsn;
			int startIdx = minSsn;
			//
			StringBuilder query = new StringBuilder();
			//
			// Merge-From Patient
			for (int idx = startIdx; idx < maxSsn; idx++)
			{
				if( !patientSsnTable.Rows.Contains(new object[] { idx }) )
				{ 
					mergeFromPatientSsn = idx;
					startIdx = idx + 1;
					break;
				}
			}
			//
			query.Append("INSERT INTO Patient (");
			query.Append("PatientGuid, VistaPatientId, PatientSsn, PatientIcn, PatientLastName,");
			query.Append("PatientFirstName, PatientMiddleName, PatientDob, PatientDobCode, PatientSexCode,");
			query.Append("LastUpdateUser, LastUpdateFunctionId, LastUpdateDate");
			query.Append(") VALUES ('");
			query.Append(returnGuids[0].ToString());
			query.Append("',");
			query.Append(mergeFromVistaId);
			query.Append(",");
			query.Append(mergeFromPatientSsn);
			query.Append(", '1010101010101010', 'FROM',");
			query.Append("'MERGE', 'A', '01/01/1949', 'B', 'M',");
			query.Append("'UNITTEST', 0, GetUtcDate()");
			query.Append(")");
		
			UnitTestUtil.RunSQL(query.ToString());
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Merge-To Patient
			for (int idx = startIdx; idx < maxSsn; idx++)
			{
				if( !patientSsnTable.Rows.Contains(new object[] { idx }) )
				{
					mergeToPatientSsn = idx;
					break;
				}
			}
			//
			if(query.Length > 0)
			{
				query.Remove(0, query.Length);
			}
			//
			query.Append("INSERT INTO Patient (");
			query.Append("PatientGuid, VistaPatientId, PatientSsn, PatientIcn, PatientLastName,");
			query.Append("PatientFirstName, PatientMiddleName, PatientDob, PatientDobCode, PatientSexCode,");
			query.Append("LastUpdateUser, LastUpdateFunctionId, LastUpdateDate");
			query.Append(") VALUES ('");
			query.Append(returnGuids[1].ToString());
			query.Append("',");
			query.Append(mergeToVistaId);
			query.Append(",");
			query.Append(mergeToPatientSsn);
			query.Append(", '0101010101010101', 'TO',");
			query.Append("'MERGE', 'A', '01/01/1952', 'B', 'M',");
			query.Append("'UNITTEST', 0, GetUtcDate()");
			query.Append(")");
			
			UnitTestUtil.RunSQL(query.ToString());
			
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Get two patient guids to use as merge-from and -to patients
			query.Remove(0, query.Length);
			query.Append("SELECT PatientGuid, VistaPatientId ");
			query.Append("FROM Patient WHERE VistaPatientId = '");
			query.Append(mergeFromVistaId);
			query.Append("'");

			DataRow fromPatientRow = UnitTestUtil.RunSQLGetDataRow(query.ToString());
			//
			query.Remove(0, query.Length);
			query.Append("SELECT PatientGuid, VistaPatientId ");
			query.Append("FROM Patient WHERE VistaPatientId = '");
			query.Append(mergeToVistaId);
			query.Append("'");

			DataRow toPatientRow = UnitTestUtil.RunSQLGetDataRow(query.ToString());
			//
			if ( fromPatientRow.IsNull("PatientGuid") || fromPatientRow.IsNull("VistaPatientId") 
				|| toPatientRow.IsNull("PatientGuid") || toPatientRow.IsNull("VistaPatientId") )
			{
				throw new Exception("Database does not contain valid data for unit tests to run");
			}
			//	
			fromVistaPatientId = fromPatientRow["VistaPatientId"].ToString();
			string toVistaPatientId = toPatientRow["VistaPatientId"].ToString();
			//
			// Now, insert this data into PatientChange
			query.Remove(0, query.Length);
			query.Append("INSERT INTO PatientChange (PatientChangeGuid, FromPatientGuid, ToPatientGuid, MergeFromVistaPatientId, ");
			query.Append("MergeToVistaPatientId, PatientChangeStatusCode, NotificationReceivedDate, LastUpdateFunctionId, LastUpdateUser, LastUpdateDate)");
			query.Append(" VALUES ('");
			query.Append(returnGuids[2].ToString());
			query.Append("','");
			query.Append(returnGuids[0].ToString()).Append("', '");
			query.Append(returnGuids[1].ToString()).Append("', '");
			query.Append(fromVistaPatientId).Append("', '");
			query.Append(toVistaPatientId).Append("', 'P', GetUtcDate(), 0, 'UNITTEST', GetUtcDate())");

			Assert.IsTrue(UnitTestUtil.RunSQL(query.ToString()) > 0);
			//
			return returnGuids;
		}

		/// <summary>
		/// Inserts merge test data into PatientChange
		/// </summary>
		/// <returns>Row change count</returns>
		private Guid [] InsertMergePatientTestData(long mergeFromVistaId, Guid mergeFromPatient)
		{		
			this.RefreshData = true;

			Guid [] returnGuids = new Guid[2];
			//
			// to-Patient
			returnGuids[0] = Guid.NewGuid();
			// Patient Change
			returnGuids[1] = Guid.NewGuid();
			//
			DataTable patientSsnTable = UnitTestUtil.RunSQLGetDataTable("SELECT DISTINCT PatientSsn FROM Patient ORDER BY PatientSsn DESC");
			DataColumn [] patientSsnKeys = new DataColumn []{ patientSsnTable.Columns[TABLES.Patient.PatientSsn] };
			patientSsnTable.PrimaryKey = patientSsnKeys;
			//
			DataRow vistaIdRow = UnitTestUtil.RunSQLGetDataRow("SELECT MAX(VistaPatientId) AS VistaPatientId FROM Patient");
			long maxVistaId = Convert.ToInt64( vistaIdRow[TABLES.Patient.VistaPatientId] );
			long mergeToVistaId = maxVistaId + 1;
			//
			int maxSsn = 999999999;
			int minSsn = 100000000;
			int mergeFromPatientSsn = minSsn;
			int mergeToPatientSsn = minSsn;
			int startIdx = minSsn;
			//
			StringBuilder query = new StringBuilder();
			//
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Merge-To Patient
			for (int idx = startIdx; idx < maxSsn; idx++)
			{
				if( !patientSsnTable.Rows.Contains(new object[] { idx }) )
				{
					mergeToPatientSsn = idx;
					break;
				}
			}
			//
			if(query.Length > 0)
			{
				query.Remove(0, query.Length);
			}
			//
			query.Append("INSERT INTO Patient (");
			query.Append("PatientGuid, VistaPatientId, PatientSsn, PatientIcn, PatientLastName,");
			query.Append("PatientFirstName, PatientMiddleName, PatientDob, PatientDobCode, PatientSexCode,");
			query.Append("LastUpdateUser, LastUpdateFunctionId, LastUpdateDate");
			query.Append(") VALUES ('");
			query.Append(returnGuids[0].ToString());
			query.Append("',");
			query.Append(mergeToVistaId);
			query.Append(",");
			query.Append(mergeToPatientSsn);
			query.Append(", '0101010101010101', 'TO',");
			query.Append("'MERGE', 'A', '01/01/1952', 'B', 'M',");
			query.Append("'UNITTEST', 0, GetUtcDate()");
			query.Append(")");
			
			UnitTestUtil.RunSQL(query.ToString());
			
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Get two patient guids to use as merge-from and -to patients
			query.Remove(0, query.Length);
			query.Append("SELECT PatientGuid, VistaPatientId ");
			query.Append("FROM Patient WHERE VistaPatientId = '");
			query.Append(mergeFromVistaId);
			query.Append("'");

			DataRow fromPatientRow = UnitTestUtil.RunSQLGetDataRow(query.ToString());
			//
			query.Remove(0, query.Length);
			query.Append("SELECT PatientGuid, VistaPatientId ");
			query.Append("FROM Patient WHERE VistaPatientId = '");
			query.Append(mergeToVistaId);
			query.Append("'");

			DataRow toPatientRow = UnitTestUtil.RunSQLGetDataRow(query.ToString());
			//
			if ( fromPatientRow.IsNull("PatientGuid") || fromPatientRow.IsNull("VistaPatientId") 
				|| toPatientRow.IsNull("PatientGuid") || toPatientRow.IsNull("VistaPatientId") )
			{
				throw new Exception("Database does not contain valid data for unit tests to run");
			}
			//	
			string fromVistaPatientId = fromPatientRow["VistaPatientId"].ToString();
			string toVistaPatientId = toPatientRow["VistaPatientId"].ToString();
			
			// Now, insert this data into PatientChange
			query.Remove(0, query.Length);
			query.Append("INSERT INTO PatientChange (PatientChangeGuid, FromPatientGuid, ToPatientGuid, MergeFromVistaPatientId, ");
			query.Append("MergeToVistaPatientId, PatientChangeStatusCode, NotificationReceivedDate, LastUpdateFunctionId, LastUpdateUser, LastUpdateDate)");
			query.Append(" VALUES ('");
			query.Append(returnGuids[1].ToString());
			query.Append("', '");
			query.Append(mergeFromPatient.ToString()).Append("', '");
			query.Append(returnGuids[0].ToString()).Append("', '");
			query.Append(fromVistaPatientId).Append("', '");
			query.Append(toVistaPatientId).Append("', 'P', GetUtcDate(), 0, 'UNITTEST', GetUtcDate())");

			Assert.IsTrue(UnitTestUtil.RunSQL(query.ToString()) > 0);
			//
			return returnGuids;
		}

		/// <summary>
		/// Clears merge test data from PatientChange
		/// </summary>
		/// <returns>Row change count</returns>
		public void ClearMergePatientTestData(Guid fromPatientGuid, Guid toPatientGuid)
		{
			StringBuilder query = new StringBuilder();
			query.Append("DELETE FROM PatientChange WHERE FromPatientGuid = '");
			query.Append(fromPatientGuid);
			query.Append("' AND ToPatientGuid = '");
			query.Append(toPatientGuid);
			query.Append("'");
			UnitTestUtil.RunSQL(query.ToString());
			//
			query.Remove(0, query.Length);
			query.Append("DELETE FROM Patient WHERE PatientGuid IN ('");
			query.Append(fromPatientGuid);
			query.Append("','");
			query.Append(toPatientGuid);
			query.Append("')");
			UnitTestUtil.RunSQL(query.ToString());
		}

		/// <summary>
		/// Clears merge test data from PatientChange
		/// </summary>
		/// <returns>Row change count</returns>
		private void ClearMergePatientTestData(Guid patientMergeGroup)
		{
			StringBuilder query = new StringBuilder();
			//
			query.Append("SELECT PatientGuid FROM Patient WHERE PatientMergeGroup = '");
			query.Append(patientMergeGroup);
			query.Append("'");
			DataTable patientGuids = UnitTestUtil.RunSQLGetDataTable(query.ToString());
			//
			for (int idx = 0; idx < patientGuids.Rows.Count; idx++)
			{
				query.Remove(0, query.Length);
				query.Append("DELETE FROM PatientChange WHERE FromPatientGuid = '");
				query.Append(((Guid)patientGuids.Rows[idx]["PatientGuid"]).ToString());
				query.Append("' OR ToPatientGuid = '");
				query.Append(((Guid)patientGuids.Rows[idx]["PatientGuid"]).ToString());
				query.Append("'");
				UnitTestUtil.RunSQL(query.ToString());
			}
			//
			query.Remove(0, query.Length);
			query.Append("DELETE FROM Patient WHERE PatientMergeGroup IS NOT NULL AND PatientMergeGroup = '");
			query.Append(patientMergeGroup);
			query.Append("'");
			UnitTestUtil.RunSQL(query.ToString());
		}

		#endregion

	}
}
#endif
